grammar SQLParser;

@header {
    import java.math.BigDecimal; 
}
@members {
          int currentParameterMarker = 0;
}

statement 
    :   selectStatement
    |   updateStatement
    |   deleteStatement
    |   insertStatement
    |   createStatement
    |   dropStatement
    |   grantStatement
    |   alterTable
    |   alterTablespace
    ;

dropStatement
    :   dropTablespace
    |   dropTable
    ;

createStatement
    :   createTablespace
    |   createTable
    ; 

alterTable
    :   ALTER TABLE schema DOT name
    (   ALTER COLUMN? identifier (SET DATA TYPE typeSpec)
    |   SET whenCondition
    )
    ;

dropTablespace
    :   DROP TABLESPACE (identifier | fileName) PURGE?
    ;

dropTable
    :   DROP TABLE schema DOT name
    ;

createTablespace
    :   CREATE READONLY? TABLESPACE identifier?
        (FILE fileName)
        ( (tablespaceDescriptionClause dfsortClause?)
        | ( FORMAT CSV csvOptions))
    ;

csvOptions
    :   (coldelOption | chardelOption | decptOption | headerOption | boolvaluesOption | csvQuoteMode | cacheSize)*;

cacheSize
    :   CACHESIZE number (KB | MB)?;

coldelOption
    :   COLDEL constant;

chardelOption
    :   CHARDEL constant;

decptOption
    :   DECPT constant;

headerOption
    :   HEADER (YES | NO);

boolvaluesOption
    :   BOOLVALUES LPAREN constant COMMA constant RPAREN;

csvQuoteMode
    :   QUOTE (ALL | STRINGS | MINIMAL | NONE);

tablespaceDescriptionClause
    :   (DESCRIPTION IN FILE fileName)
        (LANGUAGE (
            COBOL (LPAREN replacingClause (COMMA replacingClause)* RPAREN)?
          | PL1 (LPAREN replacingClause (COMMA replacingClause)* RPAREN)? )
        )
    ;

dfsortClause
    :   USE DFSORT ACCESS (MIN THRESHOLD number)? (LPAREN StringLiteral (DOT StringLiteral)* RPAREN)?
    ;

alterTablespace
    :   ALTER TABLESPACE identifier
    (   FILE fileName
    |   tablespaceDescriptionClause
    |   dfsortClause
    |   DEACTIVATE DFSORT ACCESS
    |   csvOptions
    )
    ;

replacingClause
    :   REPLACING StringLiteral BY StringLiteral
    ;

encodingSpec
    :   ENCODING StringLiteral STRICT?
    ;

fileName returns[String fileNameStr]
    :   StringLiteral { $fileNameStr = $StringLiteral.text.substring(1, $StringLiteral.text.length() - 1); }
        encodingSpec?
    ;

createTable
    :   CREATE TABLE (identifier DOT identifier)
        LPAREN (columnDef (COMMA columnDef)*) RPAREN
        IN (identifier | FILE StringLiteral)
        whenCondition?
    ;

whenCondition
    :   WHEN disjunction
    ;

columnDef
    :   identifier typeSpec (NOT NULL)? ( (USE | USES) (usesPath (IF disjunction)? | REMAINING_BYTES) )?
    |   GROUP NAME? identifier (USE | USES) usesPath LPAREN (columnDef (COMMA columnDef)*) RPAREN
    ;

usesPath
    :   StringLiteral (DOT StringLiteral)*
    ;

typeSpec
    :   ID (LPAREN (number|STAR) (COMMA number)? RPAREN)? (PATTERN StringLiteral)?
    ;

grantStatement
    :   grantTable
    |   revokeTable
    |   grantUser
    |   revokeUser
    ;

grantUser
    :   GRANT userPermission (COMMA userPermission)*
        TO userSpecs
    ;

revokeUser
    :   REVOKE userPermission (COMMA userPermission)*
        FROM userSpecs
    ;

grantTable
    :   GRANT (ALL PRIVILEGES | tablePermission (COMMA tablePermission)*)
        ON (schema DOT)? name
        TO userSpecs
    ;

revokeTable
    :   REVOKE (ALL PRIVILEGES | tablePermission (COMMA tablePermission)*)
        ON (schema DOT)? name
        FROM userSpecs
    ;

userSpecs
    :   userSpec (COMMA userSpec)*
    ;

userSpec
    :   PUBLIC | ID
    ;

userPermission
    :   SYSADM | ( CREATETS | CREATETB | DROPTS | DROPTB)
    ;

tablePermission
    :   SELECT
    |   UPDATE
    |   DELETE
    |   INSERT
    ;

updateStatement
    :   UPDATE (schema DOT)? name SET updateSpec (COMMA updateSpec)*
        whereClause?
    ;

updateSpec
    :   identifier OP_EQUALS sum
    ;

deleteStatement
    :   DELETE FROM (schema DOT)? name
        whereClause?
    ;

insertStatement
    :   INSERT INTO (schema DOT)? name
        (LPAREN identifier (COMMA identifier)* RPAREN)?
        (VALUES LPAREN sum (COMMA sum)* RPAREN)
    ;

selectStatement
    :    SELECT
        (ALL | DISTINCT)?
        (STAR | (colRef (COMMA colRef)*))
        fromClause
        whereClause?
        groupByClause?
        orderByClause?
        fetchClause?
        SEMI?
        EOF
    ;

colRef
    :   sum label?
    ;

columnFunction
    :   HEX LPAREN colRef RPAREN
    |   LENGTH LPAREN colRef RPAREN
    |   MAX LPAREN colRef (COMMA colRef)+ RPAREN
    |   MIN LPAREN colRef (COMMA colRef)+ RPAREN
    |   SUBSTR LPAREN colRef COMMA sum (COMMA sum)? RPAREN
    |   BITAND LPAREN sum COMMA sum RPAREN
    |   BITOR  LPAREN sum COMMA sum RPAREN
    |   BITXOR LPAREN sum COMMA sum RPAREN
    |   BITNOT LPAREN sum RPAREN
    |   TO_CHAR LPAREN colRef (COMMA StringLiteral)? RPAREN
    |   TO_DATE LPAREN colRef (COMMA StringLiteral)? RPAREN
    |   TO_TIME LPAREN colRef (COMMA StringLiteral)? RPAREN
    |   TO_TIMESTAMP LPAREN colRef (COMMA StringLiteral)? RPAREN
    |   STRIP LPAREN colRef (COMMA StringLiteral (COMMA StringLiteral)?)? RPAREN
    |   UPPER LPAREN colRef RPAREN
    |   LOWER LPAREN colRef RPAREN
    |   CONCAT LPAREN colRef COMMA colRef RPAREN
    |   GETVARIABLE LPAREN StringLiteral (COMMA colRef)? RPAREN
    ;

caseStatement
    :   CASE (simpleCaseStatement | complexCaseStatement)
    ;

simpleCaseStatement
    :   colRef (WHEN sum THEN sum)+ elseClause? END
    ;

complexCaseStatement
    :   (WHEN disjunction THEN sum)+ elseClause? END
    ;

elseClause
    :   ELSE sum
    ;

fromClause
    :   FROM (schema DOT)? name
    ;

whereClause
    :   WHERE disjunction
    ;

orderByClause
    :   ORDER BY orderBySpec (COMMA orderBySpec)*
    ;

fetchClause
    :   FETCH FIRST number? (ROW | ROWS) ONLY
    ;

orderBySpec
    :   identifier (ASC | DESC)?
    ;

groupByClause
    :   GROUP BY (ALL)? identifier (COMMA identifier)*
    ;

disjunction
    :	conjunction (OR disjunction)?
    ;

conjunction
    :	bool (AND conjunction)?
    ;

bool:
        NOT bool
    |   equation
    |   LPAREN disjunction RPAREN;


equation
    :   sum (
        operator sum
        |   IS NOT? NULL
        |   NOT? IN LPAREN constant (',' constant)* RPAREN
        |   NOT? LIKE StringLiteral)
    ;

sum
    :   product ((PLUS | MINUS) sum)?;

product
    :	atom ((STAR | DIVIDE) product)?;



atom
    :	MINUS? LPAREN sum RPAREN
    |   parameterMarker
    |   MINUS? column
    |   constant
    |   caseStatement
    |   columnFunction;

operator
    :   OP_EQUALS
    |   OP_NOT_EQUALS1
    |   OP_NOT_EQUALS2
    |   OP_GREATER_THAN_OR_EQUALS
    |   OP_LESS_THAN_OR_EQUALS
    |   OP_LESS
    |   OP_GREATER
    ;

constant returns[Comparable value]
    :   NULL            { $value = null; }
    |   StringLiteral   { $value = $StringLiteral.text.substring(1, $StringLiteral.text.length() - 1); }
    |   HexLiteral      { $value = $HexLiteral.text.substring(2, $HexLiteral.text.length() - 1);  }
    |   BinaryLiteral   { $value = $BinaryLiteral.text.substring(2, $BinaryLiteral.text.length() - 1); }
    |   number          { $value = new BigDecimal($number.text); }
    ;

parameterMarker returns[int no]
    :   QUESTION_MARK { $no = currentParameterMarker++; }
    ;

name    : identifier label?;
schema  : identifier;
column  : ((schema DOT)? name DOT)? identifier;
label   : AS? identifier;

identifier returns[String value]
    :   ID { $value = $ID.text.toUpperCase(); }
    |   QUOTED_ID { $value = $QUOTED_ID.text.substring(1, $QUOTED_ID.text.length() - 1).replace("\"\"", "\""); };

HexLiteral
    : 'X\'' ('0'..'9' | 'a'..'f' | 'A'..'F')+ '\''
    ;

BinaryLiteral
    : 'B\'' ('0'..'1')+ '\''
    ;

StringLiteral
    :
    '\'' ( ~('\'') | ('\'\'') )* '\''
    ;

number  : (PLUS | MINUS)? ((DIGIT+ (DOT DIGIT+)?) | (DOT DIGIT+));

OP_GREATER                  : '>';
OP_LESS                     : '<';
OP_EQUALS                   : '=';
OP_NOT_EQUALS1              : '<>';
OP_NOT_EQUALS2              : '!=';
OP_LESS_THAN_OR_EQUALS      : '<=';
OP_GREATER_THAN_OR_EQUALS   : '>=';

CREATETS:   'CREATETS';
CREATETB:   'CREATETB';
DROPTS  :   'DROPTS';
DROPTB  :   'DROPTB';
VALUES  :   'VALUES';
INTO    :   'INTO';
USES    :   'USES';
USE     :   'USE';
PUBLIC  :   'PUBLIC';
LENGTH  :   'LENGTH';
HEX     :   'HEX';
MIN     :   'MIN';
MAX     :   'MAX';
BITNOT  :   'BITNOT';
BITAND  :   'BITAND';
BITOR   :   'BITOR';
BITXOR  :   'BITXOR';
SUBSTR  :   'SUBSTR';
TO_CHAR :   'TO_CHAR';
TO_DATE :   'TO_DATE';
TO_TIME :   'TO_TIME';
TO_TIMESTAMP :   'TO_TIMESTAMP';
STRIP   :   'STRIP';
UPPER   :   'UPPER';
LOWER   :   'LOWER';
CONCAT  :   'CONCAT';
GETVARIABLE : 'GETVARIABLE';
REPLACING : 'REPLACING';
ENCODING:   'ENCODING';
END     :   'END';
SET     :   'SET';
CASE    :   'CASE';
WHEN    :   'WHEN';
THEN    :   'THEN';
ELSE    :   'ELSE';
GRANT   :   'GRANT';
REVOKE  :   'REVOKE';
REFERENCES  :   'REFERENCES';
FILE    :   'FILE';
DESCRIPTION:   'DESCRIPTION';
//PL1     :   'PL1';
PL1     :   'PL1' | 'PL/1' | 'PLI' | 'PL/I';
COBOL   :   'COBOL';
LANGUAGE:   'LANGUAGE';
CREATE  :   'CREATE';
DROP    :   'DROP';
TABLE   :   'TABLE';
TABLESPACE: 'TABLESPACE';
LIKE    :   'LIKE';
BETWEEN :   'BETWEEN';
IS      :   'IS';
NULL    :   'NULL';
NOT     :   'NOT';
ASC     :   'ASC';
DESC    :   'DESC';
ORDER   :   'ORDER';
NAME    :   'NAME';
BY      :   'BY';
GROUP   :   'GROUP';
ALL     :   'ALL';
PRIVILEGES : 'PRIVILEGES';
TO      :   'TO';
ON      :   'ON';
DISTINCT    :   'DISTINCT';
FROM    :   'FROM';
SELECT  :   'SELECT';
WHERE   :   'WHERE';
UPDATE  :   'UPDATE';
DELETE  :   'DELETE';
INSERT  :   'INSERT';
AS      :   'AS';
AND     :   'AND';
OR      :   'OR';
IN      :   'IN';
IF      :   'IF';
SYSADM  :   'SYSADM';
ALTER   :   'ALTER';
DATA    :   'DATA';
TYPE    :   'TYPE';
COLUMN  :   'COLUMN';
ACCESS  :   'ACCESS';
DFSORT  :   'DFSORT';
DEACTIVATE: 'DEACTIVATE';
THRESHOLD:'THRESHOLD';
PATTERN :   'PATTERN';
COLDEL  :   'COLDEL';
CHARDEL :   'CHARDEL';
DECPT   :   'DECPT';
BOOLVALUES: 'BOOLVALUES';
HEADER  :   'HEADER';
YES     :   'YES';
NO      :   'NO';
LINEENDINGS:'LINEENDINGS';
WINDOWS :   'WINDOWS';
UNIX    :   'UNIX';
FORMAT  :   'FORMAT';
CSV     :   'CSV';
QUOTE   :   'QUOTE';
SEMI    :   ';';
STRINGS :   'STRINGS';
MINIMAL :   'MINIMAL';
NONE    :   'NONE';
READONLY:   'READONLY';
CACHESIZE:  'CACHESIZE';
MB      :   'MB';
KB      :   'KB';
STRICT  :   'STRICT';
FETCH   :   'FETCH';
FIRST   :   'FIRST';
ROW     :   'ROW';
ROWS    :   'ROWS';
ONLY    :   'ONLY';
PURGE   :   'PURGE';
REMAINING_BYTES : 'REMAINING_BYTES';
PLUS    :   '+';
MINUS   :   '-';
DIVIDE  :   '/';
MOD     :   '%';
LPAREN  :   '(';
RPAREN  :   ')';
COMMA   :   ',';
STAR    :   '*';
DOT     :   '.';
QUESTION_MARK   :   '?';
DIGIT   :   [0-9];
QUOTED_ID :   '"' ( ~('"') | ('""') )* '"';
ID      :   ('a'..'z' | 'A'..'Z' | '_' | '!' | '$' | '@' | '/' | '%' | '#' | '*') ('a'..'z' | 'A'..'Z' | '_' | '!' | '$' | '@' | '/' | '%' | '#' | '*' | [0-9])*; 
WS      :   [ \t\r\n]+ -> skip ;
